--[[
    Author: Steve Mills (Armpit Studios)
	Contributor: Morder85 Gaming
    Mod name: Critters
    Version: 1.0.0.0
    Date: October 24 2025



1.0.0.0:
-Initial conversion from FS22 to FS25


]]

bugdebug = false;

critters = {};
critters.modDir = g_currentModDirectory;
local modDescFile = loadXMLFile("modDesc", critters.modDir.."modDesc.xml");
critters.title = getXMLString(modDescFile, "modDesc.title.en");
critters.author = getXMLString(modDescFile, "modDesc.author");
critters.version = getXMLString(modDescFile, "modDesc.version");
-- print(critters.title.." : v"..critters.version.." by "..critters.author.." main script running");

if bugdebug then
	print(critters.title.." : BUGDEBUG IS ON!!!!!!!!!!!!!!!!!!!");
end;



---------------------------------------------------------------------------------------------------------------
-- GIANTS methods
---------------------------------------------------------------------------------------------------------------

-- function critters.registerEventListeners(vehicleType)
-- 	print(critters.title.." : v"..critters.version.." by "..critters.author.." registerEventListeners called");
-- end;

function critters:loadMap(savegame)
	self.lbLastUpdate = 0;
	
	if g_currentMission:getIsClient() then
		self.cicadaSamples = {};
		self.birdSamples = {};
		self.hawkSamples = {};
		self.frogSamples = {};
		self.dogSamples = {};
		self.trainSamples = {};
		
		-- So it doesn't always use the same random sequence, use the a couple hopefully non-zero values as the seed, which should be different pretty much every time. I tried using os.clock(), but 'os' wasn't found:
		math.randomseed(g_currentMission.environment.daylight.dayStart + g_currentMission.environment.daylight.latitude);
		
		-- Useful for seeing what all the stuff is I might be able to use:
		-- print(critters.title.." : g_currentMission.environment = ");
		-- tprint(g_currentMission.environment, true);
		
		-- Not enough documentation to figure out how to create actual environment sounds:
        -- critters.xmlSoundFile = loadXMLFile("crittersSounds", critters.modDir.."sounds/crittersSounds.xml");
		-- g_soundManager:loadSampleFromXML(xmlSoundFile, baseString, "cicadasSomething", critters.modDir, self.nodeId, 0, AudioGroup.ENVIRONMENT, self.i3dMappings, self);
		-- g_soundManager:loadSoundTemplates(critters.modDir.."sounds/crittersSounds.xml");
		-- delete(xmlSoundFile);
		
		-- Also not enough documentation to figure out how to use them as audio sources:
		-- critters.oddSors = createAudioSource("reearee", critters.modDir.."sounds/NeotibicenPruinosusPruinosus_ScissorGrinder.ogg", 100000, 1, 1.0, 3);
		-- print(critters.title.." audio source = "..critters.oddSors);
		-- -- addAudioSourceSampleElement(oddSors, critters.modDir.."sounds/MegatibicenDealbatus_PlainsCicada.ogg", 0.5);
		-- setAudioSourceRandomPlayback(critters.oddSors, true);
		
		self:loadSettings();
		self:loadSounds();
		
		local delayTime = timeUntilSamplePlaysAgain(nil, self.cicadasRepeatMult);
		
		if delayTime > 0 then
			critters.cicadaTimer = Timer.new(delayTime);
			critters.cicadaTimer.name = "cicadas";
		    critters.cicadaTimer:setFinishCallback(critterTimerCallback);
			critters.cicadaTimer:start(true);
		end;
		
		delayTime = timeUntilSamplePlaysAgain(nil, self.birdsRepeatMult);
		
		if delayTime > 0 then
			critters.birdTimer = Timer.new(delayTime);
			critters.birdTimer.name = "birds";
			critters.birdTimer:setFinishCallback(critterTimerCallback);
			critters.birdTimer:start(true);
		end;
		
		delayTime = timeUntilSamplePlaysAgain(nil, self.hawksRepeatMult);
		
		if delayTime > 0 then
			critters.hawkTimer = Timer.new(delayTime);
			critters.hawkTimer.name = "hawks";
			critters.hawkTimer:setFinishCallback(critterTimerCallback);
			critters.hawkTimer:start(true);
		end;
		
		delayTime = timeUntilSamplePlaysAgain(nil, self.frogsRepeatMult);
		
		if delayTime > 0 then
			critters.frogTimer = Timer.new(delayTime);
			critters.frogTimer.name = "frogs";
			critters.frogTimer:setFinishCallback(critterTimerCallback);
			critters.frogTimer:start(true);
		end;
		
		delayTime = timeUntilSamplePlaysAgain(nil, self.dogsRepeatMult);
		
		if delayTime > 0 then
			critters.dogTimer = Timer.new(delayTime);
			critters.dogTimer.name = "dogs";
			critters.dogTimer:setFinishCallback(critterTimerCallback);
			critters.dogTimer:start(true);
		end;
		
		delayTime = timeUntilSamplePlaysAgain(nil, self.trainRepeatMult);
		
		if delayTime > 0 then
			critters.trainTimer = Timer.new(delayTime);
			critters.trainTimer.name = "trains";
			critters.trainTimer:setFinishCallback(critterTimerCallback);
			critters.trainTimer:start(true);
		end;
	end;
	
	if bugdebug then
		print(self.title.." : all sounds loaded");
	end;
end;

function critters:update()
	if g_currentMission:getIsClient() then
		
		-- It appears that g_currentMission.player is nil until after the map is loaded. Is there a postLoad method or something like that? But this works for now:
		if self.lbs == nil then
			self:makeLightningBugs();
		end;
		
		if self.lbs ~= nil then
			local timeNow = g_currentMission.time;
			
			-- I really don't need to update this any faster than 30fps. time is milliseconds, so 1000 ÷ 30 = 33.333:
			if timeNow > self.lbLastUpdate + 33.333 then
				-- Check for dead bugs:
				local playerNode = getPlayerNode();
				local num = table.getn(self.lbs);
				
				for i = num,1,-1 do
					local lb = self.lbs[i];
					local dist = 0;
					
					-- Also check for bugs too far away:
					if playerNode ~= nil then
						local x,y,z = getWorldTranslation(playerNode);
						
						dist = math.sqrt(math.pow(lb.x - x, 2) + math.pow(lb.z - z, 2));
					end;
					
					-- Only delete them when their blinker is off:
					if (lb.life >= lb.dieAt or dist > 100) and lb.alpha <= 0 then
						table.remove(self.lbs, i);
					end;
				end;
				
				-- If the current number of bugs < a random number of bugs at this moment, add one more:
				if table.getn(self.lbs) < targetNumLightningBugs() then
					if playerNode ~= nil then
						self:make1LightningBug(playerNode);
					end;
				end;
				
				for k, lb in pairs(self.lbs) do
					lb.dist = lb.dist + math.sqrt(math.pow(lb.xLen, 2) + math.pow(lb.zLen, 2));
					lb.x = lb.x + lb.xLen + math.sin(lb.dist * 5) * 0.002; -- Add a little x wiggle.
					lb.y = lb.y + lb.yLen * math.sin(lb.dist);
					lb.z = lb.z + lb.zLen + math.cos(lb.dist * 5) * 0.002; -- Add a little z wiggle.
					-- Time the bug has been alive in seconds:
					lb.life = lb.life + (timeNow - self.lbLastUpdate) / 1000;
					
					if lb.blinkWave == "sine" then
						lb.alpha = math.sin(lb.life * 8 * lb.blinkRate);
					elseif lb.blinkWave == "square" then
						local val = lb.life * lb.blinkRate;
						val = math.fmod(val, 1);
						
						if val >= 0.85 then
							lb.alpha = 1;
						else
							lb.alpha = 0;
						end;
					else
						lb.alpha = math.random();
					end;
					
					if lb.alpha < 0 then
						lb.alpha = 0;
					end;
				end;
				
				self.lbLastUpdate = timeNow;
			end;
		end;
	end;
end;


function critters:draw()
	if g_client ~= nil  then
		if self.lbs ~= nil then
			for k, lb in pairs(self.lbs) do
				local x = lb.x;
				local y = lb.y;
				local z = lb.z;
				
				-- Lightning bug color taken right from a photo: 0.909,0.949,0.596. The only problem is that drawDebugPoint is barely using those color values. They're coming out looking white. I had to go with extreme values to get a noticable yellow-green:
				drawDebugPoint(x,y,z,		0.2,1,0,		lb.alpha);
			end;
		end;
	end;
end;

function critters:mouseEvent(posX, posY, isDown, isUp, button)
	if isDown and button == 3 and Input.isKeyPressed(Input.KEY_ralt) then
		print(self.title.." : XXXXXXXXXXXXXXXXXXXXXXXXXXX temp code XXXXXXXXXXXXXXXXXXXXX");
		self.lbs = nil;
		self:makeLightningBugs();
	end;
end;

---------------------------------------------------------------------------------------------------------------
-- Custom methods
---------------------------------------------------------------------------------------------------------------

function critters:loadSettings()
	local settingsFilePath = getUserProfileAppPath().."modSettings/critters.xml";
	local someSettingWasNil = false;
	local function checkMissingValue(var, defaultVal)
		if var == nil then
			someSettingWasNil = true;
			var = defaultVal;
		end;
		
		return var;
	end;
	
	if not fileExists(settingsFilePath) then
		copyFile(critters.modDir.."settingsTemplate.xml", settingsFilePath, true);
	end;
	
	local settingsFile = loadXMLFile("crittersXML", settingsFilePath);
	
	-- I couldn't get the volumes to work, so maybe have settings for min/max time between cicadas.
	-- critters.outdoorCicadasVolume = getXMLFloat(settingsFile, "critters.outdoorCicadasVolume", 100) / 100.0;
	-- critters.indoorCicadasVolume = getXMLFloat(settingsFile, "critters.indoorCicadasVolume", 50) / 100.0;
	
	critters.cicadasRepeatMult = getXMLFloat(settingsFile, "critters.cicadasRepeatTimeMultiplier");
	critters.cicadasRepeatMult = checkMissingValue(critters.cicadasRepeatMult, 1);
	
	critters.birdsRepeatMult = getXMLFloat(settingsFile, "critters.birdsRepeatTimeMultiplier");
	critters.birdsRepeatMult = checkMissingValue(critters.birdsRepeatMult, 1);
	
	critters.hawksRepeatMult = getXMLFloat(settingsFile, "critters.hawksRepeatTimeMultiplier");
	critters.hawksRepeatMult = checkMissingValue(critters.hawksRepeatMult, 100);
	
	critters.frogsRepeatMult = getXMLFloat(settingsFile, "critters.frogsRepeatTimeMultiplier");
	critters.frogsRepeatMult = checkMissingValue(critters.frogsRepeatMult, 1);
	
	critters.dogsRepeatMult = getXMLFloat(settingsFile, "critters.dogsRepeatTimeMultiplier");
	critters.dogsRepeatMult = checkMissingValue(critters.dogsRepeatMult, 10);
	
	critters.trainRepeatMult = getXMLFloat(settingsFile, "critters.trainRepeatTimeMultiplier");
	critters.trainRepeatMult = checkMissingValue(critters.trainRepeatMult, 50);
	
	critters.lbCountMulti = getXMLFloat(settingsFile, "critters.lightningBugsCountMultiplier");
	critters.lbCountMulti = checkMissingValue(critters.lbCountMulti, 1);
	
	if someSettingWasNil then
		print(self.title.." : Some setting was missing, so writing new user settings file.");
		setXMLFloat(settingsFile, "critters.cicadasRepeatTimeMultiplier", critters.cicadasRepeatMult);
		setXMLFloat(settingsFile, "critters.birdsRepeatTimeMultiplier", critters.birdsRepeatMult);
		setXMLFloat(settingsFile, "critters.hawksRepeatTimeMultiplier", critters.hawksRepeatMult);
		setXMLFloat(settingsFile, "critters.frogsRepeatTimeMultiplier", critters.frogsRepeatMult);
		setXMLFloat(settingsFile, "critters.dogsRepeatTimeMultiplier", critters.dogsRepeatMult);
		setXMLFloat(settingsFile, "critters.trainRepeatTimeMultiplier", critters.trainRepeatMult);
		setXMLFloat(settingsFile, "critters.lightningBugsCountMultiplier", critters.lbCountMulti);
		saveXMLFile(settingsFile);
	end;
	
	delete(settingsFile);
end;

function critters:loadSounds()
	local xmlSoundFile = loadXMLFile("ciciadasSounds", critters.modDir.."sounds/crittersSounds.xml");
	
	self.cicadaSamples = {};
	self.birdSamples = {};
	self.hawkSamples = {};
	self.frogSamples = {};
	self.dogSamples = {};
	self.trainSamples = {};
	
	if xmlSoundFile ~= nil then
		local soundNode;

		if soundNode == nil then
			if g_currentMission.player ~= nil then
				soundNode = g_currentMission.player:getParentComponent();
			end;
		end;
		
		local audioGroup = AudioGroup.ENVIRONMENT;
		
		-- self.cicadaSamples.scissorGrinder = g_soundManager:loadSampleFromXML(xmlSoundFile, "sounds", "cicada1", critters.modDir, soundNode, 1, AudioGroup.ENVIRONMENT, self.i3dMappings, self);
		
		-- This lua (or GIANTS) xml stuff uses elem(index) as the xpath form for getting a 0-based indexed element named "elem". The sample name also ends up being "elem(index)".
	    local i = 0;
	    while true do
			local sam = g_soundManager:loadSample2DFromXML(xmlSoundFile, "sounds", string.format("cicada(%d)", i), critters.modDir, soundNode, 1, audioGroup);
	        if sam == nil then
	            break;
	        end;

			loadExtraSoundInfoFromXMLFile(xmlSoundFile, sam);
			table.insert(self.cicadaSamples, sam);
	        i = i + 1
	    end;
		
		i = 0;
		while true do
			local sam = g_soundManager:loadSample2DFromXML(xmlSoundFile, "sounds", string.format("bird(%d)", i), critters.modDir, soundNode, 1, audioGroup);
	        if sam == nil then
	            break;
	        end;

			loadExtraSoundInfoFromXMLFile(xmlSoundFile, sam);
			table.insert(self.birdSamples, sam);
	        i = i + 1
		end;
		
		i = 0;
		while true do
			local sam = g_soundManager:loadSample2DFromXML(xmlSoundFile, "sounds", string.format("hawk(%d)", i), critters.modDir, soundNode, 1, audioGroup);
	        if sam == nil then
	            break;
	        end;

			loadExtraSoundInfoFromXMLFile(xmlSoundFile, sam);
			table.insert(self.hawkSamples, sam);
	        i = i + 1
		end;
		
		i = 0;
		while true do
			local sam = g_soundManager:loadSample2DFromXML(xmlSoundFile, "sounds", string.format("frog(%d)", i), critters.modDir, soundNode, 1, audioGroup);
	        if sam == nil then
	            break;
	        end;

			loadExtraSoundInfoFromXMLFile(xmlSoundFile, sam);
			table.insert(self.frogSamples, sam);
	        i = i + 1
		end;
		
		i = 0;
		while true do
			local sam = g_soundManager:loadSample2DFromXML(xmlSoundFile, "sounds", string.format("dog(%d)", i), critters.modDir, soundNode, 1, audioGroup);
	        if sam == nil then
	            break;
	        end;

			loadExtraSoundInfoFromXMLFile(xmlSoundFile, sam);
			table.insert(self.dogSamples, sam);
	        i = i + 1
		end;
		
		i = 0;
		while true do
			local sam = g_soundManager:loadSample2DFromXML(xmlSoundFile, "sounds", string.format("train(%d)", i), critters.modDir, soundNode, 1, audioGroup);
	        if sam == nil then
	            break;
	        end;

			loadExtraSoundInfoFromXMLFile(xmlSoundFile, sam);
			table.insert(self.trainSamples, sam);
	        i = i + 1
		end;
		
		delete(xmlSoundFile);
	else
		print(self.title.." : crittersSounds.xml could not be opened!");
	end;
end;

function critters:onDelete()
    if self.cicadaSamples ~= nil then
        g_soundManager:deleteSamples(self.cicadaSamples);
    end;
	
	if self.birdSamples ~= nil then
		g_soundManager:deleteSamples(self.birdSamples);
	end;
	
	if self.hawkSamples ~= nil then
		g_soundManager:deleteSamples(self.hawkSamples);
	end;
	
	if self.frogSamples ~= nil then
		g_soundManager:deleteSamples(self.frogSamples);
	end;
	
	if self.dogSamples ~= nil then
		g_soundManager:deleteSamples(self.dogSamples);
	end;
	
	if self.trainSamples ~= nil then
		g_soundManager:deleteSamples(self.trainSamples);
	end;
end;

function loadExtraSoundInfoFromXMLFile(xmlSoundFile, sam)
	sam.startMonth = getXMLInt(xmlSoundFile, "sounds."..sam.sampleName..".season#startMonth", 5);
	sam.stopMonth = getXMLInt(xmlSoundFile, "sounds."..sam.sampleName..".season#stopMonth", 9);
	sam.startHour = getXMLInt(xmlSoundFile, "sounds."..sam.sampleName..".season#startHour", 10);
	sam.stopHour = getXMLInt(xmlSoundFile, "sounds."..sam.sampleName..".season#stopHour", 21);
end;

function correctMonth(month)
	-- Convert the goofy Farming Simulator month numbers where March is 1 and February is 12 to real month numbers:
	month = month + 2;
	
	if month > 12 then
		month = month - 12;
	end;
	
	return month;
end;

function critterTimerCallback(timer)
	if bugdebug then
		print(critters.title.." : timer fired");
	end;
	
	local critterList;
	local repeatMult;
	-- local isCritterSeasonFunc;
	
	if timer == critters.cicadaTimer then
		critterList = critters:cicadasForCurrentMonth();
		repeatMult = critters.cicadasRepeatMult;
		-- isCritterSeasonFunc = isCicadasTime;
	elseif timer == critters.birdTimer then
		critterList = critters.birdSamples;
		repeatMult = critters.birdsRepeatMult;
		-- isCritterSeasonFunc = isBirdsTime;
	elseif timer == critters.hawkTimer then
		critterList = critters.hawkSamples;
		repeatMult = critters.hawksRepeatMult;
		-- isCritterSeasonFunc = isBirdsTime;
	elseif timer == critters.frogTimer then
		critterList = critters.frogSamples;
		repeatMult = critters.frogsRepeatMult;
		-- isCritterSeasonFunc = isBirdsTime;
	elseif timer == critters.dogTimer then
		critterList = critters.dogSamples;
		repeatMult = critters.dogsRepeatMult;
		-- isCritterSeasonFunc = isDogsTime;
	elseif timer == critters.trainTimer then
		critterList = critters.trainSamples;
		repeatMult = critters.trainRepeatMult;
		-- isCritterSeasonFunc = isTrainTime;
	else
		print(critters.title.." : Unhandled critter timer!");
		timer:delete();
		return;
	end;
	
	local numCritters = table.getn(critterList);
	local snd = nil;

	if numCritters > 0 then
		local n = math.random(1, numCritters);
		snd = critterList[n];
		
		-- if isCritterSeasonFunc(snd) then
		if isCritterTime(snd) then
			if g_soundManager:getIsSamplePlaying(snd) == false then
				-- Set this now, so the user can change the setting at any time and the sound will use the current setting when the it starts:
				snd.volumeScale = g_gameSettings.environmentVolume;
				g_soundManager:playSample(snd);
			end;
		end;
	end;

	-- Reset to a new random time:
	timer:setDuration(timeUntilSamplePlaysAgain(snd, repeatMult));
	timer:start();
end;
			
function critters:cicadasForCurrentMonth()
	local result = {};
	local curMonth;
	
	if g_currentMission.environment.visualPeriodLocked == true then
		curMonth = correctMonth(g_currentMission.environment.currentVisualPeriod);
	else
		curMonth = correctMonth(g_currentMission.environment.currentPeriod);
	end;
	
    for k, sam in pairs(self.cicadaSamples) do
		if curMonth >= sam.startMonth and curMonth <= sam.stopMonth then
			table.insert(result, sam);
		end;
	end;
	
	return result;
end;

-- Is it the season and time when the given critter would sing?
-- @param sam: Sample loaded from sound file, which has month and time start/stop times. Can be nil.
function isCritterTime(sam)
	if bugdebug then
		print(critters.title.." : isCritterTime for sample "..sam.sampleName.." startMonth = "..sam.startMonth.." stopMonth = "..sam.stopMonth.." startHour = "..sam.startHour.." stopHour = "..sam.stopHour);
	end;
	
	local result = false;
	local curMonth;
	
	if g_currentMission.environment.visualPeriodLocked == true then
		curMonth = correctMonth(g_currentMission.environment.currentVisualPeriod);
	else
		curMonth = correctMonth(g_currentMission.environment.currentPeriod);
	end;
	
	if bugdebug then
		print(critters.title.." : curMonth = "..curMonth);
	end;
	
	if curMonth >= sam.startMonth and curMonth <= sam.stopMonth then
		-- local dlite = g_currentMission.environment.daylight;
		local curTime = g_currentMission.environment.dayTime / (1000 * 60 * 60);
		
		if bugdebug then
			print(critters.title.." : curTime = "..curTime);
		end;
		
		if sam.startHour < sam.stopHour then
			-- TODO: critterSounds.xml could include a custom element that specifies if a critter should just holler in the daylight, nighttime, or specific hours.
			if curTime >= sam.startHour and curTime <= sam.stopHour then
			-- if curTime >= dlite.dayStart and curTime <= dlite.dayEnd then
				result = true;
			end;
		else
			if curTime >= sam.startHour or curTime <= sam.stopHour then
				result = true;
			end;
		end;
	end;
	
	if bugdebug then
		print(critters.title.." : isCritterTime? "..tostring(result));
	end;
	
	return result;
end;

-- @param sam: Sample loaded from sound file, which has month and time start/stop times. Can be nil.
-- @param repeatMult: The user's multiplier to adjust how often the timer will fire.
function timeUntilSamplePlaysAgain(sam, repeatMult)
	local timerTime = 0;
	
	if repeatMult > 0 then
		if sam ~= nil then
			local curSeason;
	
			if g_currentMission.environment.visualPeriodLocked == true then
				curSeason = g_currentMission.environment.currentVisualSeason;
			else
				curSeason = g_currentMission.environment.currentSeason;
			end;
	
			if bugdebug then
				print(critters.title.." : repeatMult = "..repeatMult);
				timerTime = math.random(4, 8) * repeatMult;
			else
				-- More often during the Summer:
				-- TODO: Do I need a field in the xml for each bird that says if it's more common during the Summer? Or just let them all be that way?
				if curSeason == 1 then
					timerTime = math.random(12, 75) * repeatMult;
				else
					timerTime = math.random(20, 120) * repeatMult;
				end;
			end;
			
			-- If the sound is currently playing, add its duration to the time:
			if g_soundManager:getIsSamplePlaying(sam) then
				timerTime = timerTime + sam.duration / 1000.0;
			end;
		else
			-- Give it a minute or so, then try again:
			if bugdebug then
				timerTime = math.random(4, 8) * repeatMult;
			else
				timerTime = math.random(30, 60) * repeatMult;
			end;
		end;
	else
		-- Don't play this type of sample:
		timerTime = 0;
	end;
	
	if bugdebug then
		print(critters.title.." : timeUntilSamplePlaysAgain = "..timerTime);
	end;
	
	return timerTime * 1000;
end;

---------------------------------------------------------------------------------------------------------------
-- Lightning Bugs hunks
---------------------------------------------------------------------------------------------------------------

function critters:makeLightningBugs()
	local playerNode = getPlayerNode();
	
	if playerNode ~= nil then
		self.lbs = {};
		local n = 0;
		local num = targetNumLightningBugs();
		
		if bugdebug then
			print(self.title.." : making "..num.." lightning bugs.");
		end;
		
		repeat
			self:make1LightningBug(playerNode);
			n = n + 1;
		until n >= num;
		
		self.lbLastUpdate = 0;
	end;
end;

function critters:make1LightningBug(playerNode)
	local x,y,z = getWorldTranslation(playerNode);
	local lb = {};
	lb.x = x + (math.random(0, 200) - 100); -- Random x up to 100 away from player in either direction.
	lb.z = z + (math.random(0, 200) - 100); -- Random z up to 100 away from player in either direction.
	lb.y = getTerrainHeightAtWorldPos(g_currentMission.terrainRootNode, lb.x, 0, lb.z) + math.random(0, 1) * 0.25 + 0.1; -- Random height off ground.
	-- These Len vars are really the basis of how far the bug moves at each update.
	lb.xLen = (math.random() - 0.5) * 0.02;
	lb.zLen = (math.random() - 0.5) * 0.02;
	lb.yLen = math.random() * 0.01; -- Vertical direction shouldn't change as much as horizontals, also keep this value positive, because result of sin() will fluctuate between negative and positive.
	lb.alpha = 0; -- Use alpha to blink. Stored so we know when it's turned off - don't wanna kill bugs that are currently lit.
	lb.dist = 0; -- The total distance the bug has traveled since it was born.
	lb.life = 0; -- How long has it been alive in seconds.
	lb.dieAt = math.random(20, 60); -- Delete this bug after its life has lasted at least this long, in seconds.
	lb.blinkRate = math.random(5, 10) / 10; -- Make blink rate random between .5 and 1 second. Or at least that's what I'm shooting for.
	
	-- Some will do quick on/off blinks (square) and some will do slower fade in/out blinks (sine):
	if math.random() > 0.5 then
		lb.blinkWave = "square";
	else
		lb.blinkWave = "sine";
	end;
	
	if bugdebug then
		print(self.title.." : added lightning bug, x,z = "..lb.x..","..lb.z..", y = "..lb.y.." dieAt = "..lb.dieAt.." blinkRate = "..lb.blinkRate);
	end;
	table.insert(self.lbs, lb);
end;

function targetNumLightningBugs()
	local result = 0;
	
	-- User can turn them off in their settings file by using 0:
	if critters.lbCountMulti > 0 then
		local curMonth = 0;
	
		if g_currentMission.environment.visualPeriodLocked == true then
			curMonth = correctMonth(g_currentMission.environment.currentVisualPeriod);
		else
			curMonth = correctMonth(g_currentMission.environment.currentPeriod);
		end;
	
		-- Only during May, June, July, August, & September:
		if curMonth >= 5 and curMonth <= 9 then
			local curTime = g_currentMission.environment.dayTime / (1000 * 60 * 60);
			local dlite = g_currentMission.environment.daylight;
	
			if curTime >= dlite.nightStart - 1 or curTime <= dlite.nightEnd then
				if curTime >= dlite.nightStart or curTime <= dlite.nightEnd - 1 then
					result = 3 + math.random(0, 12); -- At least 3.
				else
					result = 1 + math.random(0, 4); -- At least 1, but not too many during twilight or dawn.
				end;
				
				result = math.ceil(result * critters.lbCountMulti);
				if bugdebug then
					print(critters.title.." : want "..result.." lightning bugs.");
				end;
			elseif bugdebug then
				print(critters.title.." : it must be day time, so no lightning bugs now.");
			end;
		elseif bugdebug then
			print(critters.title.." : month "..curMonth.." is not a lightning bug month.");
		end;
	end;
	
	return result;
end;

---------------------------------------------------------------------------------------------------------------
-- Utils
---------------------------------------------------------------------------------------------------------------

function table.contains(table, element)
	for _, value in pairs(table) do
		if value == element then
			return true
		end
	end
	return false
end

function tprint (tbl, goDeep, indent, seen)
	if not seen then seen = {}; end
	table.insert(seen, tbl);
	if not indent then indent = 0 end
	
	for k, v in pairs(tbl) do
		if k ~= nil then
			if type(k) == "table" then
				formatting = string.rep("    ", indent);
				print(formatting.."k is a table:");
				tprint(k, true);
				print(formatting..":end of table");
			else
				if type(k) == "function" then
					formatting = string.rep("    ", indent) .. tostring(k) .. ": ";
				else
					formatting = string.rep("    ", indent) .. k .. ": ";
				end;
			end;
			
			if type(v) == "table" then
				print(formatting)
				if goDeep and table.contains(seen, v) == false then
					table.insert(seen, v)
					tprint(v, true, indent+1, seen)
				end
			elseif type(v) == 'boolean' then
				print(formatting .. tostring(v))
			elseif v ~= nil then
				print(formatting .. tostring(v))
				-- print(formatting)
			else
				print(formatting .. nil);
			end
		else
			print("Some key was nil.");
		end;
	end
end

function getPlayerNode()
	local playerNode = nil;
	
	if g_currentMission.player ~= nil and g_currentMission.controlPlayer then
		playerNode = g_currentMission.player.rootNode;
	elseif g_currentMission.controlledVehicle ~= nil then
		playerNode = g_currentMission.controlledVehicle.rootNode;
	else
		playerNode = getCamera();
	end;
	
	return playerNode;
end;

addModEventListener(critters);
